<?php
/**
 * Description of StocksModel
 *
 * @author Kelvin Benard
 * @email noely@lockminds.com
 */
require_once APPPATH.'models/BMS_Model.php';

class ExpensesBudgetsModel extends BMS_Model {

    protected $branch_field = "budget_branch";
    protected $branch;
    protected $business_field = 'budget_business';
    protected $business;

    protected $filterable_date_field = 'budget_date';

    protected $table_name   = 'expense_budgets';
    protected $key          = 'budget_key';
    protected $primary_key  = 'budget_key';
    protected $soft_deletes = TRUE;
    protected $date_format  = 'datetime';
    protected $log_user     = TRUE;

    public    $selectable_fields = array('bg_deposit','bg_withdraw','budget_key','budget_status','budget_date','budget_deleted','budget_title','budget_details','budget_branch','budget_business','budget_created_by','budget_modified_by','budget_created_on','budget_modified_on','budget_deleted',
                                         'budget_deleted_by','bg_budget','bg_deposit','bg_withdraw','bg_date','bg_user','firstname','lastname',
        'expense_date','expense_details','expense_payee','expense_paid','expense_amount','expense_created_by');

    protected $set_created  = TRUE;
    protected $created_field    = 'budget_created_on';
    protected $created_by_field = 'budget_created_by';

    protected $set_modified     = TRUE;
    protected $modified_field   = 'budget_modified_on';
    protected $modified_by_field = 'budget_modified_by';

    protected $soft_delete_key    = 'budget_deleted';
    protected $deleted_by_field = 'budget_deleted_by';

    // Observers
    protected $before_insert    = array('set_key','set_ownership');
    protected $after_insert     = array();
    protected $before_update    = array();
    protected $after_update     = array();
    protected $before_find      = array('set_selectable_fields','set_join','set_select_dates','set_ownership_select','set_limit_offset');
    protected $after_find       = array();
    protected $before_delete    = array();
    protected $after_delete     = array();

    protected $return_type      = 'array';
    protected $protected_attributes   = array();

    protected $validation_rules         = array();
    protected $insert_validation_rules  = array();
    protected $skip_validation          = true;

    public function __construct(&$write_db = null, &$read_db = null)
    {
        parent::__construct($write_db, $read_db);
        $this->branch = runningBranch()->branch_key;
        $this->business = runningBusiness()->business_key;
    }

    protected function set_key($row){

        // Create a random user id between 1200 and 4294967295
        $random_unique_int = 2147483648 + mt_rand( -2147482448, 2147483647 );

        // Make sure the random user_id isn't already in use
        $query = $this->db->where( $this->key, $random_unique_int )
            ->get_where( $this->table_name );

        if( $query->num_rows() > 0 )
        {
            $query->free_result();

            // If the random user_id is already in use, try again
            return $this->taxClassKey();
        }

        if (!array_key_exists($this->key, $row))
        {
            $row[$this->key] = $random_unique_int;
        }

        return $row;
    }

    protected function set_ownership($row){

        if(!empty($this->business)){
            $row[$this->business_field] = $this->business;
        }

        if(!empty($this->branch)){
            $row[$this->branch_field] = $this->branch;
        }

        return $row;
    }

    protected function set_created_by($row)
    {
        if (!array_key_exists($this->created_by_field, $row))
        {
            $row[$this->created_by_field] = config_item("auth_user_id");
        }

        return $row;
    }

    protected function set_deleted_by($id)
    {
        $this->dbw->where($this->primary_key, $id);
        $this->db->update($this->table_name, [$this->deleted_by_field => config_item("auth_user_id")]);
    }

    protected function set_modified_by($row)
    {
        if (!array_key_exists($this->modified_by_field, $row))
        {
            $row[$this->modified_by_field] = config_item("auth_user_id");
        }

        return $row;
    }

    protected function set_modified_on($row)
    {
        if (!array_key_exists($this->modified_field, $row))
        {
            $row[$this->modified_field] = date('Y-m-d H:m:i');
        }

        return $row;
    }

    protected function set_selectable_fields()
    {
        if(!empty($this->selectable_fields)){
            $this->db->select($this->selectable_fields);
        }

    }

    protected function set_select_dates(){
        if(!empty($when = $this->input->get("when"))){
            $dates = array();
            $rawdates = datesWhenCustom($when);
            $dates['start'] = $rawdates['start'];
            $dates['end'] = $rawdates['end'];
            $this->db->where($this->filterable_date_field.'>=',$dates['start']);
            $this->db->where($this->filterable_date_field.'<=',$dates['end']);
        }
    }

    protected function set_join(){
        $this->db->join('expense_budget_transactions','expense_budget_transactions.bg_budget=expense_budgets.budget_key','left');
    }

    protected function set_ownership_select(){
        $this->db->where($this->business_field,$this->business);
        $this->db->where($this->branch_field,$this->branch);
    }

    protected function publish_event(){
        publishEventFlash('data-table-refresh');
    }

    protected function set_limit_offset(){
        if(!empty($when = $this->input->get("when"))){
            $dates = array();
            $rawdates = datesWhenCustom($when);
            $dates['start'] = $rawdates['start'];
            $dates['end'] = $rawdates['end'];
            $this->db->where($this->filterable_date_field.'>=',$dates['start']);
            $this->db->where($this->filterable_date_field.'<=',$dates['end']);
        }
    }

    function find_all($deleted = 0){

        if(!empty($when)){
            $dates = array();
            $rawdates = datesWhenCustom($when);
            $dates['start'] = $rawdates['start'];
            $dates['end'] = $rawdates['end'];
            $this->db->where('budget_date >=',$dates['start']);
            $this->db->where('budget_date <=',$dates['end']);
        }

        $this->db->select('*');
        $this->db->where('budget_branch',$this->branch);
        $this->db->where('budget_business',$this->business);
        $this->db->where('budget_deleted',$deleted);
        $this->db->select('DATE_FORMAT(budget_date, "%m/%d/%Y") as budget_date', FALSE);
        $this->db->join('expense_budget_transactions','expense_budget_transactions.bg_budget=budget_key','left');
        $this->db->select_sum('bg_deposit','total_deposit');
        $this->db->select_sum('bg_withdraw','total_withdraw');
        $this->db->group_by('budget_key');
        $query = $this->db->get('expense_budgets');
        return $query->num_rows()>0 ? $query->result_array() : false;
    }

    public function details($budget){
        $this->db->select('*');
        $this->db->where('budget_key',$budget);
        $this->db->select('DATE_FORMAT(budget_created_on, "%m/%d/%Y") as budget_date', FALSE);
        $this->db->join('expense_budget_transactions','expense_budget_transactions.bg_budget=budget_key','left');
        $this->db->select_sum('bg_deposit','total_deposit');
        $this->db->select_sum('bg_withdraw','total_withdraw');
        $this->db->group_by('budget_key');
        $query = $this->db->get("expense_budgets");
        return $query->num_rows()>0 ? $query->row_array() : false;
    }

    function deposits($budget,$when = ''){
        if(!empty($when)){
            $dates = array();
            $rawdates = datesWhenCustom($when);
            $dates['start'] = $rawdates['start'];
            $dates['end'] = $rawdates['end'];
            $this->db->where('bg_date >=',$dates['start']);
            $this->db->where('bg_date <=',$dates['end']);
        }

        $this->db->join('users_details ud','ud.user_key=bg_user','left');
        $this->db->select('*');
        $this->db->select('CONCAT_WS(" ", firstname, lastname) AS bg_user');
        $this->db->select('DATE_FORMAT(bg_date, "%m/%d/%Y") as bg_date', FALSE);
        $this->db->select('DATE_FORMAT(bg_timestamp, "%m/%d/%Y") as bg_posted_date', FALSE);
        $this->db->where('bg_budget',$budget);
        $this->db->where('bg_deposit >',0);
        $query = $this->db->get('expense_budget_transactions');
        return $query->num_rows()>0 ? $query->result_array() : false;
    }

    function withdraws($budget,$when = ''){
        if(!empty($when)){
            $dates = array();
            $rawdates = datesWhenCustom($when);
            $dates['start'] = $rawdates['start'];
            $dates['end'] = $rawdates['end'];
            $this->db->where('bg_date >=',$dates['start']);
            $this->db->where('bg_date <=',$dates['end']);
        }

        $this->db->join('users_details ud','ud.user_key=bg_user','left');
        $this->db->select('*');
        $this->db->select('CONCAT_WS(" ", firstname, lastname) AS bg_user');
        $this->db->select('DATE_FORMAT(bg_date, "%m/%d/%Y") as bg_date', FALSE);
        $this->db->select('DATE_FORMAT(bg_timestamp, "%m/%d/%Y") as bg_posted_date', FALSE);
        $this->db->where('bg_budget',$budget);
        $this->db->where('bg_withdraw >',0);
        $query = $this->db->get('expense_budget_transactions');
        return $query->num_rows()>0 ? $query->result_array() : false;
    }

    function transactions($budget,$when = ''){
        if(!empty($when)){
            $dates = array();
            $rawdates = datesWhenCustom($when);
            $dates['start'] = $rawdates['start'];
            $dates['end'] = $rawdates['end'];
            $this->db->where('bg_date >=',$dates['start']);
            $this->db->where('bg_date <=',$dates['end']);
        }

        $this->db->join('users_details ud','ud.user_key=expense_budget_transactions.bg_user','left');
        $this->db->select('*');
        $this->db->select('CONCAT_WS(" ", firstname, lastname) AS bg_user');
        $this->db->select('DATE_FORMAT(bg_date, "%m/%d/%Y") as bg_date', FALSE);
        $this->db->select('DATE_FORMAT(bg_timestamp, "%m/%d/%Y") as bg_posted_date', FALSE);
        $this->db->where('bg_budget',$budget);
        $query = $this->db->get('expense_budget_transactions');
        return $query->num_rows()>0 ? $query->result_array() : false;
    }

    function expendicture($budget,$when = ''){
        if(!empty($when)){
            $dates = array();
            $rawdates = datesWhenCustom($when);
            $dates['start'] = $rawdates['start'];
            $dates['end'] = $rawdates['end'];
            $this->db->where('expense_created_on >=',$dates['start']);
            $this->db->where('expense_created_on <=',$dates['end']);
        }

        $this->db->join('users_details ud','ud.user_key=expense_created_by','left');
        $this->db->select('*');
        $this->db->select('CONCAT_WS(" ", firstname, lastname) AS expense_created_by');
        $this->db->select('DATE_FORMAT(expense_created_on, "%m/%d/%Y") as expense_date', FALSE);
        $this->db->where('expense_budget',$budget);
        $query = $this->db->get('expenses');
        return $query->num_rows()>0 ? $query->result_array() : false;
    }

    function getBalance($budget){
        return $this->getBalanceTransactions($budget) - $this->getTotalExpenditure($budget);
    }

    function getBalanceTransactions($budget){
        $this->db->where('budget_key',$budget);
        $this->db->join('expense_budget_transactions','expense_budget_transactions.bg_budget=budget_key','left');
        $this->db->select_sum('bg_deposit','total_deposit');
        $this->db->select_sum('bg_withdraw','total_withdraw');
        $this->db->group_by('budget_key');
        $query = $this->db->get("expense_budgets");
        $transBalance =  $query->num_rows()>0 ? $query->row()->total_deposit - $query->row()->total_withdraw : 0;
        return $transBalance;
    }

    function getTotalExpenditure($budget){
        $this->db->select('SUM(DISTINCT expense_amount) as total');
        $this->db->where('expense_budget',$budget);
        $query = $this->db->get('expenses');
        return $query->num_rows()>0 ? $query->row()->total : 0;
    }

    function budgets($when = ''){
        if(!empty($when)){
            $dates = array();
            $rawdates = datesWhenCustom($when);
            $dates['start'] = $rawdates['start'];
            $dates['end'] = $rawdates['end'];
            $this->db->where('budget_created_at >=',$dates['start']);
            $this->db->where('budget_created_at <=',$dates['end']);
        }
        $this->db->select('*');
        $this->db->where('budget_branch',$this->branch);
        $this->db->select('DATE_FORMAT(budget_created_on, "%m/%d/%Y") as budget_date', FALSE);
        $this->db->join('expense_budget_transactions','expense_budget_transactions.bg_budget=expense_budgets.budget_key','left');
        $this->db->select_sum('bg_deposit','total_deposit');
        $this->db->select_sum('bg_withdraw','total_withdraw');
        $this->db->group_by('budget_key');
        $query = $this->db->get('expense_budgets');
        return $query->num_rows()>0 ? $query->result_array() : false;
    }

}
